/* batch_td_stroke_from_csv.jsx - LEAGUE-ONLY, PNG SEQUENCE, ROBUST FILTERS, EXACT FOLDER STRUCTURE */

(function () {
  // --- tiny utils ---
  function env(k, d){ var v=$.getenv(k); return (v===null||v===undefined||v==="")?d:v; }
  function fail(msg){ alert(msg); throw new Error(msg); }
  function logLine(s){ try{ var f=File(Folder.userData.fsName+"/td_batch.log"); f.open("a"); f.writeln(new Date().toISOString()+"  "+s); f.close(); }catch(e){} }

  // ASCII trim helper (for ExtendScript)
  function trim(str) {
    var s = str || "";
    while(s.charAt(0)==' ' || s.charAt(0)=='\t' || s.charAt(0)=='\n' || s.charAt(0)=='\r') { s = s.substring(1); }
    while(s.length>0 && (s.charAt(s.length-1)==' ' || s.charAt(s.length-1)=='\t' || s.charAt(s.length-1)=='\n' || s.charAt(s.length-1)=='\r')) { s = s.substring(0, s.length-1); }
    return s;
  }

  // Clean value: remove quotes, normalize unicode spaces, collapse/trim
  function cleanValue(v){
    if (!v && v !== 0) return "";
    var s = String(v);
    s = s.replace(/^["']+|["']+$/g, ''); // strip wrapping quotes
    // normalize a bunch of unicode whitespace to normal space
    s = s.replace(/[\u00A0\u1680\u180E\u2000-\u200B\u202F\u205F\u3000]/g, ' ');
    s = s.replace(/\s+/g, ' ');
    s = s.replace(/^[ \t\r\n]+|[ \t\r\n]+$/g, '');
    return s;
  }

  // --- config from env (LEAGUE-ONLY) ---
  var PROJECT   = env("AE_PROJECT", null);
  var CSV_PATH  = env("AE_CSV", null);
  var COMP_NAME = env("AE_COMP","Comp 1");
  var LAYER_NAME= env("AE_LAYER","TOUCHDOWN");
  var LEAGUE    = env("AE_LEAGUE","");   // REQUIRED
  var COLORSET  = env("AE_COLORSET","secondary").toLowerCase();
  var STROKE_W  = env("AE_STROKE_WIDTH","");
  var STROKE_OV = env("AE_STROKE_OVER","");
  var OUTDIR    = env("AE_OUTDIR","");
  var PATH_TPL  = env("AE_PATH_TEMPLATE","{league}");
  var ANIM_NAME = env("AE_ANIM","TOUCHDOWN");
  var RS_TPL    = env("AE_RS_TEMPLATE","Best Settings");
  var OM_TPL    = env("AE_OM_TEMPLATE","PNG Sequence"); // Output Module template name
  var EXT       = env("AE_EXT","_");
  var LIMIT_STR = env("AE_LIMIT","");
  var LIMIT     = LIMIT_STR ? parseInt(LIMIT_STR,10) : null;
  var QUIT_APP  = env("AE_QUIT","1")==="1";

  // --- helpers ---
  function openRead(path){ var f=new File(path); if(!f.exists) fail("File not found: "+path); f.open("r"); var s=f.read(); f.close(); return s; }
  function parseTable(txt){
    var lines=[], tmpLines=txt.split('\n');
    for(var x=0;x<tmpLines.length;x++){
      var ln=tmpLines[x]; ln=ln.replace('\r',''); if(ln) lines.push(ln);
    }
    var rows=[], i, line;
    for(i=0;i<lines.length;i++){
      line=lines[i]; if(!line) continue;
      var cells=(line.indexOf("\t")!==-1? line.split("\t"): line.split(","));
      for(var c=0;c<cells.length;c++){
        var cell=cells[c];
        // trim + dequote
        cell = trim(cell);
        if(cell.charAt(0)=='"' && cell.charAt(cell.length-1)=='"'){ cell=cell.substring(1,cell.length-1); }
        cells[c]=cell;
      }
      rows.push(cells);
    }
    if(rows.length<2) fail("CSV has no data.");
    return rows;
  }
  function headerIdx(h){
    var m={}; for(var i=0;i<h.length;i++) m[h[i].toLowerCase()]=i;
    function need(x){ if(m[x]===undefined) fail("Missing column: "+x); }
    need("abbreviation"); need("r"); need("g"); need("b"); need("r2"); need("g2"); need("b2");
    // league column is optional in header; if absent, we’ll use AE_LEAGUE for all
    return m;
  }
  function clamp01(v){ return Math.max(0, Math.min(1, v)); }
  function rgb01(r,g,b){ return [clamp01(parseInt(r,10)/255), clamp01(parseInt(g,10)/255), clamp01(parseInt(b,10)/255)]; }

  // Treat near-black (<= ~13/255) as black
  function isNearlyBlack(c){ if (!c || c.length !== 3) return true; return (c[0] <= 0.05 && c[1] <= 0.05 && c[2] <= 0.05); }
  // Prefer preferred; if black, fall back to alternate; if both black → white
  function pickSmartColor(preferred, alternate){
    var p = preferred  || [0,0,0];
    var a = alternate  || [0,0,0];
    if (!isNearlyBlack(p)) return p;
    if (!isNearlyBlack(a)) return a;
    return [1,1,1];
  }

  function buildRows(rows){
    var h=rows[0], idx=headerIdx(h), out=[], hasLeague=(idx["league"]!==undefined);
    for(var i=1;i<rows.length;i++){
      var r=rows[i];
      var ab = cleanValue(r[idx["abbreviation"]]);
      if(!ab) continue;
      var leagueVal = hasLeague ? cleanValue(r[idx["league"]]) : cleanValue(LEAGUE);
      out.push({
        abbr: ab,
        league: leagueVal,
        primary: rgb01(r[idx["r"]],r[idx["g"]],r[idx["b"]]),
        secondary: rgb01(r[idx["r2"]],r[idx["g2"]],r[idx["b2"]])
      });
    }
    return out;
  }

  // LEAGUE-ONLY filtering
  function pickTeamsLeagueOnly(all){
    var res=[], i;
    var targetLeague = cleanValue(LEAGUE).toUpperCase();
    if (!targetLeague) fail("AE_LEAGUE is required for league-only mode.");
    for (i=0; i<all.length; i++){
      var teamLeague = cleanValue(all[i].league).toUpperCase();
      if (teamLeague === targetLeague) res.push(all[i]);
    }
    if (LIMIT && res.length > LIMIT) res = res.slice(0, LIMIT);
    return res;
  }

  function findComp(name){ for(var i=1;i<=app.project.numItems;i++){ var it=app.project.item(i); if(it instanceof CompItem && it.name===name) return it; } return null; }

  function setTextStroke(layer, color, widthOpt, overOpt){
    var st=layer.property("Source Text"); if(!st) fail("Layer has no Source Text: "+layer.name);
    var td=st.value; td.applyStroke=true; td.strokeColor=color;
    if(widthOpt!=="" && widthOpt!==null && widthOpt!==undefined){ var w=parseFloat(widthOpt); if(!isNaN(w)) td.strokeWidth=w; }
    if(overOpt!=="" && overOpt!==null && overOpt!==undefined){ td.strokeOverFill=(overOpt==="1"||overOpt===true); }
    st.setValue(td);
  }

  // LOOP-BASED SANITIZE (no regex) for file/folder names
  function sanitize(s){
    var t=s||"";
    var bad="\\/:*?\"<>|";
    var result="";
    for(var i=0;i<t.length;i++){
      var ch=t.charAt(i);
      var isBad=false;
      for(var j=0;j<bad.length;j++){
        if(ch==bad.charAt(j)){ isBad=true; break; }
      }
      result += (isBad ? "-" : ch);
    }
    return result;
  }

  // Ensure animation name doesn't start with separators so we avoid double "__"
  function animNameSafe(s){
    var v = cleanValue(s);
    // strip any leading separators/underscores/hyphens/spaces
    v = v.replace(/^[\s_\-]+/, "");
    return v;
  }

  // --- run ---
  if (app.beginSuppressDialogs) { try { app.beginSuppressDialogs(); } catch(e) {} }
  app.beginUndoGroup("Batch Stroke From CSV - League Only");

  if(!PROJECT) fail("AE_PROJECT env not set.");
  var projFile=new File(PROJECT); if(!projFile.exists) fail("AE_PROJECT not found: "+PROJECT);
  app.open(projFile);

  if(!CSV_PATH) fail("AE_CSV env not set.");

  if(!LEAGUE || cleanValue(LEAGUE)==="") fail("AE_LEAGUE is required.");

  var csvText=openRead(CSV_PATH);
  var table=parseTable(csvText);
  var allRows=buildRows(table);
  var targets=pickTeamsLeagueOnly(allRows);
  if(!targets.length) fail("No teams matched league: " + LEAGUE);

  var comp=findComp(COMP_NAME); if(!comp) fail("Comp not found: "+COMP_NAME);
  var layer=comp.layer(LAYER_NAME); if(!layer) fail("Layer not found: "+LAYER_NAME);

  var rootOut = OUTDIR ? new Folder(OUTDIR) : (app.project.file ? app.project.file.parent : Folder.desktop);
  if(!rootOut.exists) rootOut.create();

  // Clear any existing render queue items
  while(app.project.renderQueue.numItems > 0) { app.project.renderQueue.item(1).remove(); }

  // Preview
  var teamNames = [];
  for (var i=0; i<Math.min(10, targets.length); i++) { teamNames.push(targets[i].abbr + "(" + targets[i].league + ")"); }
  if (targets.length > 10) teamNames.push("...");
  alert("LEAGUE-ONLY: About to render " + targets.length + " teams for '" + cleanValue(LEAGUE).toUpperCase() + "'. First few: " + teamNames.join(", "));

  var rendered=0;
  var ANIM_SAFE = animNameSafe(ANIM_NAME);

  for(var i=0;i<targets.length;i++){
    var t = targets[i];
    var strokeColor = (COLORSET === "primary") ? pickSmartColor(t.primary,  t.secondary) : pickSmartColor(t.secondary, t.primary);
    setTextStroke(layer, strokeColor, STROKE_W, STROKE_OV);

    // League + (optional) conference parsing (kept for NCAAF etc.)
    var leagueLabel = t.league || LEAGUE || "NA";
    var confLabel = t.conference || "";

    if (!confLabel && leagueLabel.indexOf("_") !== -1) {
      var parts = leagueLabel.split("_");
      leagueLabel = parts[0];
      confLabel = parts.slice(1).join("_");
    }

    // Path: OUTDIR/{league}/
    var sub = PATH_TPL.replace("{league}", sanitize(leagueLabel)).replace("{abbr}", sanitize(t.abbr));
    var outFolder = new Folder(rootOut.fsName + "/" + sub);
    if(!outFolder.exists) outFolder.create();

    // Anim folder:
    // with conference:   NCAAF_ACC_FLA_G_1_6
    // without conference: NHL_NHL_UTAH_G_1_6  (duplicate league)
    var animFolderName;
    if (confLabel) {
      animFolderName = sanitize(leagueLabel) + "_" + sanitize(confLabel) + "_" + sanitize(t.abbr) + "_" + ANIM_SAFE;
    } else {
      animFolderName = sanitize(leagueLabel) + "_" + sanitize(leagueLabel) + "_" + sanitize(t.abbr) + "_" + ANIM_SAFE;
    }
    var animFolder = new Folder(outFolder.fsName + "/" + animFolderName);
    if (!animFolder.exists) animFolder.create();

    // Output file for PNG sequence — OM template will add numbering.
    var outFile = File(animFolder.fsName + "/0000");

    var rq = app.project.renderQueue.items.add(comp);
    try{ rq.applyTemplate(RS_TPL); }catch(e){}
    var om = rq.outputModule(1);
    try{ om.applyTemplate(OM_TPL); }catch(e){}
    om.file = outFile;

    logLine("Setting up render -> " + animFolder.fsName);
    rendered++;
  }

  // Render all at once
  if(rendered > 0) {
    logLine("Starting batch render of " + rendered + " teams...");
    app.project.renderQueue.render();
  }

  app.endUndoGroup();
  if (app.endSuppressDialogs) { try { app.endSuppressDialogs(); } catch(e) {} }
  logLine("Done: rendered "+rendered+" animation(s).");
  if(QUIT_APP) app.quit();
})();
